/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.utils;

import android.Manifest;
import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build.VERSION;
import android.os.Environment;
import android.os.storage.StorageManager;
import android.provider.DocumentsContract;
import android.provider.Settings;

import androidx.documentfile.provider.DocumentFile;

import com.foxit.uiextensions.R;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.UUID;

public class AppStorageManager {
    public static final int NAME_MAX = 255;

    private Context mContext;
    private Map<String, Boolean> mCacheMap;
    private static AppStorageManager mAppStorageManager = null;
    private WeakReference<Activity> mActivity;

    public static final String DIRECTORY_SCOPED_CACHE  = "ScopedCache";

    public static AppStorageManager getInstance(Context context) {
        if (mAppStorageManager == null) {
            mAppStorageManager = new AppStorageManager(context.getApplicationContext());
        }
        if (context instanceof Activity){
            mAppStorageManager.mActivity = new WeakReference<>((Activity) context);
        }
        return mAppStorageManager;
    }

    private AppStorageManager(Context context) {
        mContext = context;
        if (VERSION.SDK_INT >= 19) {
            mCacheMap = new HashMap<String, Boolean>(5);
        }
    }

    public File getCacheDir() {
        return mContext.getCacheDir();
    }

    public boolean checkStorageCanWrite(String filePath) {
        if (VERSION.SDK_INT < 19) {
            return true;
        }
        if (filePath.startsWith(Environment.getExternalStorageDirectory().getPath())) {
            return true;
        }
        List<String> list = getVolumePaths();
        boolean result = false;
        for (String path : list) {
            if (filePath.startsWith(path)) {
                if (mCacheMap != null && mCacheMap.containsKey(path)) {
                    result = mCacheMap.get(path);
                    break;
                }
                File file = new File(path, ".foxit-" + UUID.randomUUID());
                if (file.exists()) {
                    file.delete();
                }
                result = file.mkdir();
                if (result) {
                    file.delete();
                }
                if (mCacheMap != null) {
                    mCacheMap.put(path, result);
                }
                break;
            }
        }
        return result;
    }

    @TargetApi(14)
    private List<String> getVolumePathsAboveVersion14() {
        List<String> result = null;
        try {
            StorageManager storageManager = (StorageManager) mContext.getSystemService(Context.STORAGE_SERVICE);
            Method getPathsMethod = storageManager.getClass().getMethod("getVolumePaths");
            Method getVolumeStateMethod = storageManager.getClass().getMethod("getVolumeState", String.class);
            String[] paths = (String[]) getPathsMethod.invoke(storageManager);
            result = new ArrayList<String>();
            for (String path : paths) {
                String state = (String) getVolumeStateMethod.invoke(storageManager, path);
                if (Environment.MEDIA_MOUNTED.equals(state)) {
                    result.add(path);
                }
            }
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
        return result;
    }

    public List<String> getVolumePaths() {
        if (VERSION.SDK_INT >= 14) {
            List<String> volumList = getVolumePathsAboveVersion14();
            if (volumList != null && volumList.size() > 0) {
                return volumList;
            }
        }
        List<String> volumList = new ArrayList<String>();
        String sdCard = Environment.getExternalStorageDirectory().getPath();
        if (sdCard != null) {
            volumList.add(sdCard);
        }
        File mountsFile = new File("/proc/mounts");
        if (mountsFile.exists()) {
            Scanner scanner;
            try {
                scanner = new Scanner(mountsFile);
                while (scanner.hasNext()) {
                    String line = scanner.nextLine();
                    if (line.startsWith("/dev/block/vold/")) {
                        String[] lineElements = line.split(" ");
                        String element = lineElements[1];
                        if (!volumList.contains(element)) {
                            File f = new File(element);
                            if (f.exists() && f.isDirectory()) {
                                volumList.add(element);
                            }
                        }
                    }
                }
                scanner.close();
            } catch (FileNotFoundException e) {
                e.printStackTrace();
            }
        }
        File voldFile = new File("/system/etc/vold.fstab");
        if (voldFile.exists()) {
            Scanner scanner;
            try {
                scanner = new Scanner(mountsFile, "UTF-8");
                while (scanner.hasNext()) {
                    String line = scanner.nextLine();
                    if (line.startsWith("/dev/block/vold/")) {
                        String[] lineElements = line.split(" ");
                        String element = lineElements[1];
                        if (!volumList.contains(element)) {
                            File f = new File(element);
                            if (f.exists() && f.isDirectory()) {
                                volumList.add(element);
                            }
                        }
                    }
                }
                scanner.close();
            } catch (FileNotFoundException e) {
                e.printStackTrace();
            }
        }
        return volumList;
    }

    // feature//reader//b
    public boolean needManageExternalStoragePermission() {
        boolean needPermission = false;
        try {
            PackageInfo packageInfo = mContext.getPackageManager().getPackageInfo(mContext.getPackageName(), PackageManager.GET_PERMISSIONS);
            for (String requestedPermission : packageInfo.requestedPermissions) {
                if (requestedPermission.contentEquals(Manifest.permission.MANAGE_EXTERNAL_STORAGE)){
                    needPermission = true;
                    break;
                }
            }
        }catch (Exception ignored){
        }
        return needPermission;
    }

    private final String SP_KEY_DEFAULT_FOLDER = "sp_key_default_folder";

    public void setDefaultFolder(String path){
        AppSharedPreferences.getInstance(mContext).setString(mContext.getPackageName(),SP_KEY_DEFAULT_FOLDER, path);
    }

    public String getDefaultFolder(){
        return AppSharedPreferences.getInstance(mContext).getString(mContext.getPackageName(),SP_KEY_DEFAULT_FOLDER,"");
    }

    private static int mOpenTreeRequestCode = 0xF001;

    public static void setOpenTreeRequestCode(int requestCode){
        mOpenTreeRequestCode = requestCode;
    }

    public static int getOpenTreeRequestCode(){
        return mOpenTreeRequestCode;
    }

    public boolean checkCallDocumentTreeUriPermission(String path, boolean excludeRoot) {
        if (excludeRoot && isRootVolumePath(path)){
            return true;
        }
        if (mActivity.get() == null){
            return false;
        }
        return AppFileUtil.checkCallDocumentTreeUriPermission(mActivity.get(),getOpenTreeRequestCode(),
                AppFileUtil.toDocumentUriFromPath(path));
    }


    public boolean copyDocument(String sourcePath, String targetPath, boolean cleanSource) {
        boolean result = false;
        Uri uri = AppFileUtil.toDocumentUriFromPath(targetPath);
        if (!DocumentsContract.isDocumentUri(mContext, uri)){
            return false;
        }
        DocumentFile parentDocument = getExistingDocumentFile(
                AppFileUtil.toDocumentUriFromPath(targetPath.substring(0,targetPath.lastIndexOf("/"))));
        if (parentDocument == null)return false;
        DocumentFile documentFile = getExistingDocumentFile(uri);
        Uri outUri = uri;
        String mimeTye = AppFileUtil.getMimeType(uri.getPath());
        String displayName = AppFileUtil.getFileName(uri.getPath());
        if (documentFile == null){
            documentFile = parentDocument.createFile(mimeTye, displayName);
        }else {
            documentFile = parentDocument.createFile(mimeTye, System.currentTimeMillis() + displayName);
            if (documentFile != null && documentFile.exists()){
                outUri = documentFile.getUri();
            }
        }
        try (InputStream inputStream = new FileInputStream(sourcePath);
             OutputStream outputStream = mContext.getContentResolver().openOutputStream(outUri)) {
            AppFileUtil.copy(inputStream, outputStream);
            if (outUri != uri){
                DocumentFile oldFile =  parentDocument.findFile(displayName);
                if (oldFile != null){
                    if (oldFile.delete()){
                        if (!documentFile.renameTo(displayName)){
                            documentFile.delete();
                        }
                    }
                }
            }
            if (cleanSource){
                File file = new File(sourcePath);
                if (file.exists()){
                    //noinspection ResultOfMethodCallIgnored
                    file.delete();
                }
            }
            result = true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    public DocumentFile getExistingDocumentFile(Uri uri){
        DocumentFile documentFile = null;
        if (!AppFileUtil.existsDocumentFile(mContext, uri))return null;
        if (AppFileUtil.isDocumentTreeUri(uri)){
            documentFile = findExternalStorageDocument(mContext, uri);
        }else if (DocumentFile.isDocumentUri(mContext, uri)){
            documentFile = findExternalStorageDocument(mContext, uri);
        }
        return documentFile;
    }

    private static DocumentFile findExternalStorageDocument(Context context, Uri uri) {
        DocumentFile documentFile = null;
        List<String> pathSegments;
        if (uri != null && AppFileUtil.isExternalStorageDocument(uri)){
            pathSegments = uri.getPathSegments();
            if (pathSegments != null && pathSegments.size() == 4){
                String[] subPathSegments = null;
                String treePath = pathSegments.get(1);
                String documentPath = pathSegments.get(3);
                if (!treePath.equals(documentPath)){
                    subPathSegments = documentPath.substring(treePath.length()+1).split("/");
                }
                documentFile = DocumentFile.fromTreeUri(context,uri);
                if (documentFile != null && documentFile.getUri().equals(uri)) {
                    String uriStr = uri.toString();
                    String parentPath = uriStr.substring(0, uriStr.lastIndexOf(Uri.encode(pathSegments.get(2))));
                    documentFile = DocumentFile.fromTreeUri(context, Uri.parse(parentPath));
                }
                if (subPathSegments != null){
                    for (String path : subPathSegments) {
                        if (documentFile != null){
                            documentFile = documentFile.findFile(path);
                        }
                    }
                }
            }else if (pathSegments != null){
                documentFile = DocumentFile.fromTreeUri(context,uri);
                if (documentFile != null && documentFile.getUri().equals(uri)) {
                    String uriStr = uri.toString();
                    String parentPath = uriStr.substring(0, uriStr.lastIndexOf(Uri.encode(pathSegments.get(2))));
                    documentFile = DocumentFile.fromTreeUri(context, Uri.parse(parentPath));
                }
            }
        }
        return documentFile;
    }

    public boolean checkDirectoryPermission(String path) {
        AppFileUtil.updateIsExternalStorageManager();
        if (!AppFileUtil.needScopedStorageAdaptation())
            return true;
        if (AppFileUtil.existsDocumentFile(mContext, AppFileUtil.toDocumentUriFromPath(path))){
            return true;
        }
        UIToast.getInstance(mContext).show(R.string.failed_to_select_target_folder_toast);
        return false;
    }

    public boolean isEquivalentExternalFilePath(String path, String otherPath) {
        if (path == null && otherPath == null)return true;
        boolean result = false;
        if (path != null)
            result = path.equals(otherPath);
        if (!result && (path != null && otherPath != null)){
            result = getExternalRelativePath(path).equals(getExternalRelativePath(otherPath));
        }
        return result;
    }

    public String getExternalRelativePath(String path){
        String externalRootDirectory = startWithVolumePath(path);
        if (externalRootDirectory != null){
            path = path.substring(externalRootDirectory.length());
        }else if (path != null && path.startsWith(getScopedCacheDir())){
            path = path.substring(getScopedCacheDir().length());
        }
        return path;
    }

    public String startWithVolumePath(String path){
        String volumePath = null;
        for (String vPath : getVolumePaths()) {
            if (path.startsWith(vPath)){
                volumePath = vPath;
                break;
            }
        }
        return volumePath;
    }

    public boolean isRootVolumePath(String path){
        return getVolumePaths().contains(path);
    }

    public String getScopedCacheDir(){
        return getCacheDir() + File.separator + DIRECTORY_SCOPED_CACHE;
    }

    public String toExternalPathFromScopedCache(String path) {
        String result = null;
        if (path != null && path.startsWith(getScopedCacheDir())){
            result = AppFileUtil.getSDPath() + getExternalRelativePath(path);
        }
        return result;
    }

    public String getScopedCacheFilePath(String path) {
        return getScopedCacheDir() + getExternalRelativePath(path);
    }

    // feature//reader//b
    public static boolean isExternalStorageManager() {
        if (VERSION.SDK_INT >= 30)
            return Environment.isExternalStorageManager();
        return false;
    }

    // feature//reader//b
    public void requestExternalStorageManager(Activity activity, int requestCode) {
        if (VERSION.SDK_INT < 30 || activity == null)return;
        Intent intent = new Intent(Settings.ACTION_MANAGE_APP_ALL_FILES_ACCESS_PERMISSION);
        intent.setData(Uri.parse("package:" + activity.getPackageName()));
        activity.startActivityForResult(intent, requestCode);
    }
}
