/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.utils;


import static android.os.Build.VERSION.SDK_INT;

import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.view.DisplayCutout;
import android.view.View;
import android.view.Window;
import android.view.WindowInsets;
import android.view.WindowManager;

import androidx.core.view.ViewCompat;
import androidx.core.view.WindowCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.core.view.WindowInsetsControllerCompat;

public class SystemUiHelper {
    private static SystemUiHelper instance;

    private boolean mbSystemUIShown = false;
    private boolean mbFullScreen = false;
    private boolean mbAllowedHideSystemUI = true;

    private SystemUiHelper() {
    }

    public static SystemUiHelper getInstance() {
        if (instance == null)
            instance = new SystemUiHelper();
        return instance;
    }

    public int getStatusBarHeight(Context context) {
        int height = 0;
        int resourceId = context.getResources().getIdentifier("status_bar_height", "dimen", "android");
        if (resourceId > 0) {
            height = context.getResources().getDimensionPixelSize(resourceId);
        }
        return height;
    }

    //https://developer.android.com/training/system-ui/immersive.html?hl=zh-cn#java
    public void hideSystemUI(Activity activity) {
        if (activity == null) return;
        if (!mbAllowedHideSystemUI) return;

        mbSystemUIShown = false;
        hideSystemUI(activity.getWindow());
    }

    public void hideSystemUI(Window window) {
        if (window == null) {
            return;
        }

        if (SDK_INT >= 30) {
            WindowInsetsControllerCompat windowInsetsController =
                    WindowCompat.getInsetsController(window, window.getDecorView());
            // Hide the system bars.
            windowInsetsController.hide(WindowInsetsCompat.Type.statusBars());
        } else {
            // Enables regular immersive mode.
            // For "lean back" mode, remove SYSTEM_UI_FLAG_IMMERSIVE.
            // Or for "sticky immersive," replace it with SYSTEM_UI_FLAG_IMMERSIVE_STICKY
            window.getDecorView().setSystemUiVisibility(
                    View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY
                            // Set the content to appear under the system bars so that the
                            // content doesn't resize when the system bars hide and show.
                            | View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                            | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                            | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                            // Hide the nav bar and status bar
                            | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                            | View.SYSTEM_UI_FLAG_FULLSCREEN);
        }
    }

    // Shows the system bars by removing all the flags
    // except for the ones that make the content appear under the system bars.
    public void showSystemUI(Activity activity) {
        if (activity == null) return;
        mbSystemUIShown = true;
        Window window = activity.getWindow();
        if (SDK_INT >= 30) {
            if (isFullScreenMode(activity)) {
                WindowInsetsControllerCompat windowInsetsController =
                        WindowCompat.getInsetsController(window, window.getDecorView());
                // Hide the system bars.
                windowInsetsController.show(WindowInsetsCompat.Type.navigationBars());
            } else {
                WindowInsetsControllerCompat windowInsetsController =
                        WindowCompat.getInsetsController(window, window.getDecorView());
                // Hide the system bars.
                windowInsetsController.show(WindowInsetsCompat.Type.systemBars());
            }
        } else {
            activity.getWindow().getDecorView().setSystemUiVisibility(
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                            | View.SYSTEM_UI_FLAG_VISIBLE
                    /*| View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN*/);
        }
    }

    public void showNavigationBar(Activity activity) {
        if (activity == null) return;
        showNavigationBar(activity.getWindow());
    }

    public void showNavigationBar(Window window) {
        if (window == null) return;
        if (SDK_INT >= 30) {
            WindowInsetsControllerCompat windowInsetsController =
                    WindowCompat.getInsetsController(window, window.getDecorView());
            // Hide the system bars.
            windowInsetsController.show(WindowInsetsCompat.Type.navigationBars());
        } else {
            View view = window.getDecorView();
            int oldFlags = view.getSystemUiVisibility();
            int newFlags = oldFlags;

            newFlags &= ~(View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                    | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION);
            if (newFlags != oldFlags) {
                view.setSystemUiVisibility(newFlags);
                view.requestLayout();
            }
        }
    }

    public void hideNavigationBar(Activity activity) {
        if (activity == null) return;
        hideNavigationBar(activity.getWindow());
    }

    public void hideNavigationBar(Window window) {
        if (window == null) return;
        if (SDK_INT >= 30) {
            WindowInsetsControllerCompat windowInsetsController =
                    WindowCompat.getInsetsController(window, window.getDecorView());
            // Hide the system bars.
            windowInsetsController.hide(WindowInsetsCompat.Type.navigationBars());
        } else {
            View view = window.getDecorView();

            int oldFlags = view.getSystemUiVisibility();
            int newFlags = oldFlags;
            newFlags |= (View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                    | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION);
            if (newFlags != oldFlags) {
                view.setSystemUiVisibility(newFlags);
                view.requestLayout();
            }
        }
    }

    public void hideStatusBar(Activity activity) {
        if (activity == null) return;

        hideStatusBar(activity.getWindow());
    }

    public void hideStatusBar(Window window) {
        if (window == null) return;

        if (SDK_INT >= 30) {
            WindowInsetsControllerCompat windowInsetsController =
                    WindowCompat.getInsetsController(window, window.getDecorView());
            // Hide the system bars.
            windowInsetsController.hide(WindowInsetsCompat.Type.statusBars());
        } else {
            View view = window.getDecorView();
            int oldFlags = view.getSystemUiVisibility();
            int newFlags = oldFlags;

//            newFlags &= ~(View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
//                    | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION);
            newFlags |= View.SYSTEM_UI_FLAG_FULLSCREEN;
            if (newFlags != oldFlags) {
                view.setSystemUiVisibility(newFlags);
                view.requestLayout();
            }
        }
    }

    public void showStatusBar(Activity activity) {
        if (activity == null) return;
        showStatusBar(activity.getWindow());
    }

    public void showStatusBar(Window window) {
        if (window == null) return;

        if (SDK_INT >= 30) {
            WindowInsetsControllerCompat windowInsetsController =
                    WindowCompat.getInsetsController(window, window.getDecorView());
            // Hide the system bars.
            windowInsetsController.show(WindowInsetsCompat.Type.statusBars());
        } else {
            View view = window.getDecorView();
            int oldFlags = view.getSystemUiVisibility();
            int newFlags = oldFlags;

            newFlags &= ~(View.SYSTEM_UI_FLAG_FULLSCREEN);
//            newFlags |= (View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
//                    | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION);
            if (newFlags != oldFlags) {
                view.setSystemUiVisibility(newFlags);
                view.requestLayout();
            }
        }
    }

    public boolean isSystemUIShown() {
        return mbSystemUIShown;
    }

    public boolean isStatusBarShown(Activity context) {
        if (context == null) return false;
        if (!mbSystemUIShown) return false;

        if (this.isFullScreenMode(context)) {
            return false;
        }

        if (Build.VERSION.SDK_INT >= 30) {
            WindowInsets insets = context.getWindow().getDecorView().getRootWindowInsets();
            if (insets != null) {
                return insets.isVisible(WindowInsets.Type.statusBars()) || insets.isVisible(WindowInsets.Type.displayCutout());
            }
        }
        return false;
    }

    public boolean isCutoutVisible(Activity context) {
        if (context == null) return false;
        if (!mbSystemUIShown) return false;

        if (SDK_INT >= 28) {
            WindowInsets insets = context.getWindow().getDecorView().getRootWindowInsets();
            if (insets != null) {
                if (Build.VERSION.SDK_INT >= 30) {
                    return insets.isVisible(WindowInsets.Type.displayCutout());
                } else {
                    DisplayCutout cutout = insets.getDisplayCutout();
                    return  cutout != null;
                }
            }
        }
        return false;
    }

    public boolean isFullScreenMode(Activity context) {
        if (context == null) return false;
        return isFullScreenMode(context.getWindow());
    }

    public boolean isFullScreenMode(Window window) {
        if (window == null) return false;

        WindowManager.LayoutParams params = window.getAttributes();
        int paramsFlag = params.flags & WindowManager.LayoutParams.FLAG_FULLSCREEN;
        return paramsFlag == WindowManager.LayoutParams.FLAG_FULLSCREEN;
    }

    public boolean isFullScreen() {
        return mbFullScreen;
    }

    public void setFullScreen(boolean fullScreen) {
        this.mbFullScreen = fullScreen;
    }

    public void setAllowedHideSystemUI(Activity activity, boolean allowed) {
        if (activity == null) return;

        mbAllowedHideSystemUI = allowed;
        if (isFullScreen()) {
            if (allowed)
                hideSystemUI(activity);
            else
                showSystemUI(activity);
        }
    }

    public boolean isAllowedHideSystemUI() {
        return mbAllowedHideSystemUI;
    }

    public void setStatusBarColor(Window window, int statusColor) {
        if (window == null) return;

        int targetSdkVersion = window.getContext().getApplicationInfo().targetSdkVersion;
        if (targetSdkVersion >= 35 && Build.VERSION.SDK_INT >= 35) {
            WindowInsetsControllerCompat windowInsetsController =
                    ViewCompat.getWindowInsetsController(window.getDecorView());
            if (windowInsetsController == null) {
                return;
            }
            windowInsetsController.setAppearanceLightStatusBars(false);
        } else {
            window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
                window.setStatusBarColor(statusColor);
            }
            window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_VISIBLE);
        }
    }

    public void setNotFocusable(Window window) {
        if (window == null) return;
        window.setFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE, WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE);
    }

    public void clearNotFocusable(Window window) {
        if (window == null) return;
        window.clearFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE);
    }
}
