/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.function;

import android.content.Context;
import android.graphics.Bitmap;
import android.os.Environment;
import android.text.format.Time;
import android.widget.Toast;

import com.foxit.pdf.function_demo.R;
import com.foxit.sdk.PDFException;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;
import java.util.UUID;

public class Common {
    public static final int ANNOTATION = 0;
    public static final int OUTLINE = 1;
    public static final int DOCINFO = 2;
    public static final int PDF_TO_TXT = 3;
    public static final int PDF_TO_IMAGE = 4;
    public static final int IMAGE_TO_PDF = 5;
    public static final int SIGNATURE = 6;
    public static final int WATERMARK = 7;
    public static final int SEARCH = 8;
    public static final int GRAPHICS_OBJECTS = 9;

    private static final String inputFiles = "input_files.txt";

    private static String externalPath;

    public static String getSuccessInfo(Context context, String path) {
        return context.getString(R.string.fx_file_saved_successd, path);
    }

    public static String getExternalPath() {
        return externalPath;
    }

    public static void setExternalPath(String path) {
        externalPath = path;
    }

    public static boolean checkSD() {
        boolean sdExist = Environment.getExternalStorageState().equals(android.os.Environment.MEDIA_MOUNTED);
        if (sdExist) {
            File sddir = Environment.getExternalStorageDirectory();
            setExternalPath(sddir.getPath());
        } else {
            setExternalPath(null);
        }
        return sdExist;
    }

    public static String getFixFolder() {
        String path = getExternalPath();
        path += "/input_files/";
        return path;
    }

    public static boolean createFolder(String folderPath) {
        try {
            File myFilePath = new File(folderPath);
            if (!myFilePath.exists()) {
                myFilePath.mkdirs();
            }
        } catch (Exception e) {
        }
        return true;
    }

    public static String getOutputFilesFolder(int type) {
        //Combine the current external path, outputting files path (fixed) and example module name together
        String outputPath = getExternalPath();
        outputPath += "/output_files/";

        String moduleName = getModuleName(type);
        if (moduleName != null && moduleName.trim().length() > 1)
            outputPath += moduleName + "/";
        createFolder(outputPath);
        return outputPath;
    }

    public static boolean saveImageFile(Bitmap bitmap, Bitmap.CompressFormat picFormat, String fileName) {
        File file = new File(fileName);
        try {
            FileOutputStream fos = new FileOutputStream(file);
            bitmap.compress(picFormat, 100, fos);
            fos.flush();
            fos.close();
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }

        return true;
    }

    public static PDFDoc loadPDFDoc(Context context, String path, byte[] password) {
        try {
            PDFDoc doc = new PDFDoc(path);
            if (doc.isEmpty()) {
                Toast.makeText(context, context.getString(R.string.fx_the_path_not_exist_error, path), Toast.LENGTH_LONG).show();
                return null;
            }

            doc.load(password);
            return doc;
        } catch (PDFException e) {
            Toast.makeText(context, context.getString(R.string.fx_load_document_error, e.getMessage()), Toast.LENGTH_LONG).show();
        }
        return null;
    }

    public static PDFPage loadPage(Context context, PDFDoc doc, int index, int parseFlag) {
        PDFPage page = null;
        if (doc == null || doc.isEmpty()) {
            Toast.makeText(context, context.getString(R.string.fx_the_document_is_null), Toast.LENGTH_LONG).show();
            return page;
        }

        try {
            page = doc.getPage(index);
            if (page == null || page.isEmpty()) {
                Toast.makeText(context, context.getString(R.string.fx_the_page_is_null), Toast.LENGTH_LONG).show();
                return page;
            }

            if (!page.isParsed()) {
                Progressive progressive = page.startParse(parseFlag, null, false);

                int state = Progressive.e_ToBeContinued;
                while (state == Progressive.e_ToBeContinued) {
                    state = progressive.resume();
                }

                if (state == Progressive.e_Error) {
                    Toast.makeText(context, context.getString(R.string.fx_parse_page_error), Toast.LENGTH_LONG).show();
                    return null;
                }
            }

        } catch (PDFException e) {
            Toast.makeText(context, context.getString(R.string.fx_load_page_error, e.getMessage()), Toast.LENGTH_LONG).show();
        }
        return page;
    }

    public static boolean saveDFDoc(Context context, PDFDoc doc, String save_path) {
        try {
            boolean ret = doc.saveAs(save_path, PDFDoc.e_SaveFlagNoOriginal);
            if (ret) {
                Toast.makeText(context, getSuccessInfo(context, save_path), Toast.LENGTH_LONG).show();
                return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        Toast.makeText(context, context.getString(R.string.fx_save_doc_error), Toast.LENGTH_LONG).show();
        return false;
    }

    public static DateTime getCurrentDateTime() {
        Time now = new Time();
        now.setToNow();

        DateTime dateTime = null;

        int year = now.year;
        int month = now.month + 1;
        int date = now.monthDay;
        int hour = now.hour;
        int minute = now.minute;
        int second = now.second;
        int timezone = TimeZone.getDefault().getRawOffset();
        int localHour = timezone / 3600000;
        int localMinute = timezone % 3600000 / 60;

        dateTime = new DateTime();
        dateTime.set(year, month, date, hour, minute, second, 0, (short) localHour, localMinute);

        return dateTime;
    }

    public static String randomUUID(String separator) {
        String uuid = UUID.randomUUID().toString();
        if (separator != null) {
            uuid.replace("-", separator);
        }
        return uuid;
    }

    //Check whether the SD is available.
    public static boolean isSDAvailable() {
        return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
    }

    public static String getSDPath() {
        return Environment.getExternalStorageDirectory().getPath();
    }

    private static boolean exist(String path) {
        File file = new File(path);
        return file.exists();
    }

    private static boolean mergeFiles(Context context, String outDir, List<String> files) {
        boolean success = false;
        OutputStream os = null;
        try {

            byte[] buffer = new byte[1 << 13];
            for (String f : files) {
                File outFile = new File(outDir + f);
                createParentPath(outFile);
                if (exist(getFixFolder() + f))
                    continue;

                os = new FileOutputStream(outDir + f);
                InputStream is = context.getAssets().open(f);
                int len = is.read(buffer);
                while (len != -1) {
                    os.write(buffer, 0, len);
                    len = is.read(buffer);
                }
                is.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                if (os != null) {
                    os.flush();
                    os.close();
                    success = true;
                }
            } catch (IOException ignore) {
            }
        }
        return success;
    }

    private static void createParentPath(File file) {
        File parentFile = file.getParentFile();
        if (null != parentFile && !parentFile.exists()) {
            parentFile.mkdirs();
            createParentPath(parentFile);
        }
    }

    public static void copyTestFiles(Context context) {
        if (Common.isSDAvailable()) {
            List<String> testFiles = getAssetsList(context);
            mergeFiles(context, getFixFolder(), testFiles);
        }
    }

    private static List<String> getAssetsList(Context context) {
        List<String> files = new ArrayList<>();
        InputStream inputStream = null;
        BufferedReader br = null;
        try {
            inputStream = context.getAssets().open(new File(inputFiles).getPath());
            br = new BufferedReader(new InputStreamReader(inputStream));
            String path;
            while (null != (path = br.readLine())) {
                files.add(path);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                if (inputStream != null)
                    inputStream.close();
                if (br != null)
                    br.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return files;
    }

    public static String getFileNameWithoutExt(String filePath) {
        int index = filePath.lastIndexOf('/');
        String name = filePath.substring(index + 1);
        index = name.lastIndexOf('.');
        if (index > 0) {
            name = name.substring(0, index);
        }
        return name;
    }

    private static String getModuleName(int type) {
        String name = "";
        switch (type) {
            case ANNOTATION:
                name = "annotation";
                break;
            case OUTLINE:
                name = "outline";
                break;
            case DOCINFO:
                name = "docInfo";
                break;
            case PDF_TO_TXT:
                name = "pdf2text";
                break;
            case PDF_TO_IMAGE:
                name = "render";
                break;
            case IMAGE_TO_PDF:
                name = "image2pdf";
                break;
            case SIGNATURE:
                name = "signature";
                break;
            case WATERMARK:
                name = "watermark";
                break;
            case SEARCH:
                name = "search";
                break;
            case GRAPHICS_OBJECTS:
                name = "graphics_objects";
                break;
            default:
                name = null;
                break;
        }
        return name;
    }

    public static void checkDirectoryAvailable(String path) throws PDFException {
        if (path == null){
            throw new PDFException(Constants.e_ErrFile, "The output directory can't be null!");
        }
        File file = new File(path);
        if ((!file.exists() && !file.mkdirs() || (file.exists() && !file.canWrite()))){
            throw new PDFException(Constants.e_ErrFile, "The output directory is unavailable, " +
                    "please check write permission to it!");
        }
    }
}

