/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.function;


import android.content.Context;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.common.fxcrt.RectF;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.TextSearch;

import java.io.FileWriter;
import java.io.IOException;

public class Search {
    private Context mContext;

    public Search(Context context) {
        this.mContext = context;
    }

    public void startSearch() {
        String inputPath = Common.getFixFolder() + "AboutFoxit.pdf";
        String outputPath = Common.getOutputFilesFolder(Common.SEARCH) + "search.txt";

        PDFDoc doc = Common.loadPDFDoc(mContext, inputPath, null);
        if (doc == null) return;

        FileWriter text_out = null;
        try {
            text_out = new FileWriter(outputPath, false);

            // sample 1: search for all pages of doc.
            TextSearch search = new TextSearch(doc, null, TextPage.e_ParseTextNormal);

            int start_index = 0, end_index = doc.getPageCount() - 1;
            search.setStartPage(0);
            search.setEndPage(doc.getPageCount() - 1);

            String pattern = "Foxit";
            search.setPattern(pattern,false);

            int flags = TextSearch.e_SearchNormal;
            // If want to specify flags, you can do as followings:
            // flags |= TextSearch::e_SearchMatchCase;
            // flags |= TextSearch::e_SearchMatchWholeWord;
            // flags |= TextSearch::e_SearchConsecutive;
            search.setSearchFlags(flags);
            wrapperWrite(text_out, "Begin search " + pattern + " at " + inputPath + ".\n");
            wrapperWrite(text_out, "Start index:\t" + start_index + "\r\n");
            wrapperWrite(text_out, "End index:\t" + end_index + "\r\n");
            wrapperWrite(text_out, "Match key:\t" + pattern + "\r\n");
            String match_case = ((flags & TextSearch.e_SearchMatchCase) != 0) ? "Yes" : "No";
            wrapperWrite(text_out, "Match Case\t" + match_case + "\r\n");
            String match_whole_word = ((flags & TextSearch.e_SearchMatchWholeWord) != 0) ? "Yes" : "No";
            wrapperWrite(text_out, "Match whole word:\t" + match_whole_word + "\r\n");
            String match_consecutive = ((flags & TextSearch.e_SearchConsecutive) != 0) ? "Yes" : "No";
            wrapperWrite(text_out, "Consecutive:\t" + match_consecutive + "\r\n");
            int match_count = 0;
            while (search.findNext()) {
                outputMatchedInfo(text_out, search, match_count);
                match_count++;
            }
            text_out.close();
            Toast.makeText(mContext, Common.getSuccessInfo(mContext, outputPath), Toast.LENGTH_LONG).show();
        } catch (Exception e) {
            Toast.makeText(mContext, e.getMessage(), Toast.LENGTH_LONG).show();
        }
    }

    private void wrapperWrite(FileWriter text_out, String formatbuff) throws IOException {
        text_out.write(formatbuff);
    }

    private void outputMatchedInfo(FileWriter text_out, TextSearch search, int matched_index) throws PDFException, IOException {
        int page_index = search.getMatchPageIndex();
        wrapperWrite(text_out, "Index of matched pattern:\t" + matched_index + "\r\n");
        wrapperWrite(text_out, "\tpage:\t" + page_index + "\r\n");
        wrapperWrite(text_out, "\tmatch char start index:\t" + search.getMatchStartCharIndex() + "\r\n");
        wrapperWrite(text_out, "\tmatch char end index:\t" + search.getMatchEndCharIndex() + "\r\n");
        wrapperWrite(text_out, "\tmatch sentence start index:\t" + search.getMatchSentenceStartIndex() + "\r\n");
        wrapperWrite(text_out, "\tmatch sentence:\t" + search.getMatchSentence() + "\r\n");
        RectFArray rect_array = search.getMatchRects();
        int rect_count = rect_array.getSize();
        wrapperWrite(text_out, "\tmatch rectangles count:\t" + rect_count + "\r\n");
        for (int i = 0; i < rect_count; i++) {
            RectF rect = rect_array.getAt(i);
            wrapperWrite(text_out, String.format("\trectangle(in PDF space) :%d\t[left = %.4f, bottom = %.4f, right = %.4f, top = %.4f]\r\n", i, rect.getLeft(), rect.getBottom(), rect.getRight(), rect.getTop()));
        }
    }

}
