/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.function.annotation;

import android.content.Context;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.IconProviderCallback;
import com.foxit.sdk.pdf.annots.ShadingColor;

import java.io.InputStream;
import java.util.HashMap;

import static com.foxit.sdk.common.Constants.e_ErrSuccess;

class CustomIconProvider extends IconProviderCallback {
    private HashMap<String, PDFDoc> mStampDocMap;
    private boolean mIsDynamicStamp;
    private Context mContext;

    CustomIconProvider(Context context) {
        mStampDocMap = new HashMap<>();
        mIsDynamicStamp = false;
        mContext = context;
    }

    @Override
    public void release() {

    }

    // If one icon provider offers different style icon for one icon name of a kind of annotaion,
    // please use different provider ID or version in order to distinguish different style for Foxit PDF SDK.
    // Otherwise, only the first style icon for the same icon name of same kind of annotation will have effect.
    @Override
    public String getProviderID() {
        if (mIsDynamicStamp) {
            return "Simple Demo Dynamic IconProvider";
        } else {
            return "Simple Demo IconProvider";
        }
    }

    @Override
    public String getProviderVersion() {
        return "1.0.0";
    }

    @Override
    public boolean hasIcon(int annot_type, String icon_name) {
        if (annot_type != Annot.e_Stamp) return false;

        String assetsPath;
        if (mIsDynamicStamp) {
            assetsPath = "DynamicStamps/" + icon_name + ".pdf";
        } else {
            assetsPath = "StaticStamps/" + icon_name + ".pdf";
        }

        try {
            PDFDoc doc = mStampDocMap.get(assetsPath);
            if (doc != null && !doc.isEmpty()) {
                return doc.getPageCount() >= 1;
            }

            InputStream is = mContext.getAssets().open(assetsPath);
            int length = is.available();
            byte[] buffer = new byte[length];
            is.read(buffer);

            doc = new PDFDoc(buffer);
            int error_code = doc.load(null);
            if (e_ErrSuccess == error_code) {
                mStampDocMap.put(assetsPath, doc);
            }
            is.close();
            return !doc.isEmpty();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean canChangeColor(int annot_type, String icon_name) {
        return false;
    }

    @Override
    public PDFPage getIcon(int annot_type, String icon_name, int color) {
        if (annot_type != Annot.e_Stamp) return null;

        String assetsPath;
        if (mIsDynamicStamp) {
            assetsPath = "DynamicStamps/" + icon_name + ".pdf";
        } else {
            assetsPath = "StaticStamps/" + icon_name + ".pdf";
        }

        try {
            PDFDoc doc = mStampDocMap.get(assetsPath);
            if (doc == null || doc.isEmpty() || doc.getPageCount() < 1)
                return null;
            return doc.getPage(0);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public boolean getShadingColor(int annot_type, String icon_name, int referenced_color, int shading_index, ShadingColor out_shading_color) {
        return false;
    }

    @Override
    public float getDisplayWidth(int annot_type, String icon_name) {
        return 0.0f;
    }

    @Override
    public float getDisplayHeight(int annot_type, String icon_name) {
        return 0.0f;
    }

    void setUseDynamicStamp(boolean isDynamicStamp) {
        mIsDynamicStamp = isDynamicStamp;
    }
}
