/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.main;

import android.Manifest;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.provider.Settings;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.DividerItemDecoration;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.pdf.function.Common;
import com.foxit.pdf.function.DocInfo;
import com.foxit.pdf.function.GraphicsObjects;
import com.foxit.pdf.function.Image2Pdf;
import com.foxit.pdf.function.Outline;
import com.foxit.pdf.function.Pdf2text;
import com.foxit.pdf.function.Render;
import com.foxit.pdf.function.Search;
import com.foxit.pdf.function.Signature;
import com.foxit.pdf.function.Watermark;
import com.foxit.pdf.function.annotation.Annotation;
import com.foxit.pdf.function_demo.R;
import com.foxit.pdf.main.FunctionAdapter.FunctionItemBean;
import com.foxit.pdf.main.FunctionAdapter.OnItemClickListener;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Library;

import java.util.ArrayList;
import java.util.List;

public class MainActivity extends FragmentActivity {
    private static final String sn = "aKEpCRKu0/BkCmRFhavCoXOP9xI8QwJzIAEe4+CuldzMNJGzf4mKvQ==";
    private static final String key = "ezJvjt3GpGp399sXJUWUjY5Q4Cy/RvybylNKusfzMufW5k1ziSNqlwk6IKRg++yKjh3VA44Jx0t3zTngokyX78qqdQX2n9Dq3DEulcv/f5XN1B6EX2jESCT4K53PBEncOAqvgam7qmIxuVj0yIJMcVzRju9BID1xQKOFHiRSUUXupPrEKTraVBJgzsphlgudAsQAk3xS56wpFAEAMJE9RhgHlzLd758GK5viOEglYUHl7n31UilbrvpOKiGEfQ52D54IwAofHdHamOMSlSYXhh2/JXWNKi+EMyegfB26Qg04+sry01n6SqLofbxC+6WtcRuwEj2I1Yb0lc6yFESdJWPGv7Dgz0COyGQbM6GDTNE33vaeFIa4MQdnKZuXABj9kGmTjeiZbcy8vkY7Lmm0gmnVu09VIpqZhYZwJM/aYdBRMPxBnKDGVw1t7hLD8jeOJAZX1Ua1jKjsLcni7s5+W7WU0sGKr7DJTwsEzcLqyr160onAaU3/M8ZQs7wy43rZ//oPYpx/mZHJs927TP7oVeRHx0sffTGpKPN7lctjA2hStQTkhCPl7nZhEz7c5EC3t9GJ+z4l5Sf+XLDILXyVTBkfDX897CFt1ozEU46y5TokHjpufDXSZxxBGcLxOiMfbW2MzfiB/BTL6/rbn/6kYM4aoGxhP4xWYEcVTpipDTzXlJ062+f4FEFoEt1YS5ufFA2v2nc9pqAzMpkciPXID6Wn822/PW/8cOu/bzVC1CH9lCG6YQtDV0PQyrjiOWkRQMN19N1Mfwooay7gA59NL01iSNYn5qx8R0O3oH2UfFBde3wCSqo+08ZvLnVPV0HJjacNHsuRlQvnGmBJvkNoeL/zb4O5t15dFBCrBUGRdlqTQl22r1n3sTNVoCkQfCGtY5wI/UbsCRWsjxphWBa0Edh1COD62UeY7Gq6oWy7O5b4kRjXVfMfc0q848KgUxfB03QFw7H40lHXODQOyKGfcCGMNdvA8zYRqSgg35DH9pmDqDB2rFyohlXy1fR+3e/Fud5tdlosppauZ/6rbFPiM5k4IAPQGRs0P/hW9MCeMZR8xnhcXO6Dabfnc7cdfkZqt+CI/+U0Jy+piWwCTnq97r5hba1Ng6eaCuvK9kzirBX1+HyQ+Fc6PEnIdi/lQNm7XjLy7aoWiUuNYoCFLHwdxV/XgGNyNALGz+HmHpESRMjjOjlhPIahkl9ENAZKW3Oq+k+K6d/fsht4ACz4sWSgTmiY+odH3OiwZ8wIhio0onqea9F9sHfWtHUcP+UYgxjVrH+/b32KhWaxwrtKiIg=";

    private static final int REQUEST_EXTERNAL_STORAGE = 1000;
    private static final int REQUEST_ALL_FILES_ACCESS_PERMISSION = 2000;
    private static final String[] PERMISSIONS_STORAGE = {
            Manifest.permission.READ_EXTERNAL_STORAGE,
            Manifest.permission.WRITE_EXTERNAL_STORAGE
    };

    private int initErrCode = Constants.e_ErrSuccess;
    private boolean isPermissionDenied = true;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        if (!Common.checkSD()) {
            Toast.makeText(MainActivity.this, getString(R.string.fx_the_sdcard_not_exist), Toast.LENGTH_LONG).show();
            return;
        }

        checkPermission();
        initLibrary();
        initView();
    }

    private void checkPermission() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            if (!Environment.isExternalStorageManager()) {
                Intent intent = new Intent(Settings.ACTION_MANAGE_APP_ALL_FILES_ACCESS_PERMISSION);
                intent.setData(Uri.parse("package:" + getApplicationContext().getPackageName()));
                startActivityForResult(intent, REQUEST_ALL_FILES_ACCESS_PERMISSION);
            } else {
                onPermissionGranted();
            }
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            int permission = ContextCompat.checkSelfPermission(this.getApplicationContext(), Manifest.permission.WRITE_EXTERNAL_STORAGE);
            if (permission != PackageManager.PERMISSION_GRANTED) {
                ActivityCompat.requestPermissions(this, PERMISSIONS_STORAGE, REQUEST_EXTERNAL_STORAGE);
            } else {
                onPermissionGranted();
            }
        } else {
            onPermissionGranted();
        }
    }

    private void initLibrary() {
        initErrCode = Library.initialize(sn, key);
        showLibraryErrorInfo();
    }


    private void onPermissionGranted() {
        isPermissionDenied = false;
        Common.copyTestFiles(getApplicationContext());
    }

    private void showLibraryErrorInfo() {
        if (initErrCode != Constants.e_ErrSuccess) {
            if (initErrCode == Constants.e_ErrInvalidLicense) {
                Toast.makeText(getApplicationContext(), getStr(R.string.fx_the_license_is_invalid), Toast.LENGTH_LONG).show();
            } else {
                Toast.makeText(getApplicationContext(), getStr(R.string.fx_failed_to_initalize), Toast.LENGTH_LONG).show();
            }
        }
    }

    private void initView() {
        RecyclerView recyclerView = findViewById(R.id.function_list);
        FunctionAdapter adapter = new FunctionAdapter(getApplicationContext(), getFunctionItems());
        recyclerView.setAdapter(adapter);
        recyclerView.setLayoutManager(new LinearLayoutManager(getApplicationContext(), LinearLayoutManager.VERTICAL, false));
        recyclerView.setItemAnimator(new DefaultItemAnimator());
        recyclerView.addItemDecoration(new DividerItemDecoration(getApplicationContext(), DividerItemDecoration.VERTICAL));
        adapter.setOnItemClickListener(new OnItemClickListener() {
            @Override
            public void onItemClick(int positon, FunctionItemBean itemBean) {
                if (initErrCode != Constants.e_ErrSuccess) {
                    showLibraryErrorInfo();
                    return;
                }

                if (isPermissionDenied) {
                    Toast.makeText(getApplicationContext(), getString(R.string.fx_permission_denied), Toast.LENGTH_LONG).show();
                    return;
                }

                int type = itemBean.type;
                switch (type) {
                    case Common.ANNOTATION:
                        Annotation annotation = new Annotation(getApplicationContext());
                        annotation.addAnnotation();
                        break;
                    case Common.OUTLINE:
                        Outline outline = new Outline(getApplicationContext());
                        outline.modifyOutline();
                        break;
                    case Common.DOCINFO:
                        DocInfo info = new DocInfo(getApplicationContext());
                        info.outputDocInfo();
                        break;
                    case Common.PDF_TO_TXT:
                        Pdf2text pdf2text = new Pdf2text(getApplicationContext());
                        pdf2text.doPdfToText();
                        break;
                    case Common.PDF_TO_IMAGE:
                        Render render = new Render(getApplicationContext());
                        render.renderPage(0);
                        break;
                    case Common.IMAGE_TO_PDF:
                        Image2Pdf image2Pdf = new Image2Pdf(getApplicationContext());
                        image2Pdf.doImage2Pdf();
                        break;
                    case Common.SIGNATURE:
                        Signature signature = new Signature(getApplicationContext());
                        signature.addSignature(0);
                        break;
                    case Common.WATERMARK:
                        Watermark watermark = new Watermark(getApplicationContext());
                        watermark.addWatermark();
                        break;
                    case Common.SEARCH:
                        Search search = new Search(getApplicationContext());
                        search.startSearch();
                        break;
                    case Common.GRAPHICS_OBJECTS:
                        GraphicsObjects graphicsObjects = new GraphicsObjects(getApplicationContext());
                        graphicsObjects.addGraphicsObjects();
                        break;
                    default:
                        break;
                }
            }
        });
    }

    private List<FunctionItemBean> getFunctionItems() {
        List<FunctionItemBean> functions = new ArrayList<>();
        //pdf2txt
        FunctionItemBean pdf2txt = new FunctionItemBean(Common.PDF_TO_TXT, getStr(R.string.pdf2text), getStr(R.string.pdf2textInfo));
        functions.add(pdf2txt);
        //outline
        FunctionItemBean outline = new FunctionItemBean(Common.OUTLINE, getStr(R.string.outline), getStr(R.string.outlineInfo));
        functions.add(outline);
        //annotation
        FunctionItemBean annotation = new FunctionItemBean(Common.ANNOTATION, getStr(R.string.addAnnotation), getStr(R.string.addAnnotationInfo));
        functions.add(annotation);
        //docinfo
        FunctionItemBean docInfo = new FunctionItemBean(Common.DOCINFO, getStr(R.string.docInfo), getStr(R.string.docInfoDemoInfo));
        functions.add(docInfo);
        //pdf2image
        FunctionItemBean pdf2image = new FunctionItemBean(Common.PDF_TO_IMAGE, getStr(R.string.render), getStr(R.string.renderInfo));
        functions.add(pdf2image);
        //signature
        FunctionItemBean signature = new FunctionItemBean(Common.SIGNATURE, getStr(R.string.signature), getStr(R.string.signatureInfo));
        functions.add(signature);
        //image2pdf
        FunctionItemBean image2pdf = new FunctionItemBean(Common.IMAGE_TO_PDF, getStr(R.string.image2pdf), getStr(R.string.image2pdfInfo));
        functions.add(image2pdf);
        //watermark
        FunctionItemBean watermark = new FunctionItemBean(Common.WATERMARK, getStr(R.string.watermark), getStr(R.string.watermarkInfo));
        functions.add(watermark);
        //search
        FunctionItemBean search = new FunctionItemBean(Common.SEARCH, getStr(R.string.search), getStr(R.string.searchInfo));
        functions.add(search);
        //graphics
        FunctionItemBean graphics = new FunctionItemBean(Common.GRAPHICS_OBJECTS, getStr(R.string.graphics), getStr(R.string.graphicsInfo));
        functions.add(graphics);
        return functions;
    }

    private String getStr(int resId) {
        return getApplicationContext().getString(resId);
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == REQUEST_EXTERNAL_STORAGE) {
            if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                onPermissionGranted();
            } else {
                isPermissionDenied = true;
            }
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == REQUEST_ALL_FILES_ACCESS_PERMISSION) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                if (Environment.isExternalStorageManager()) {
                    onPermissionGranted();
                } else {
                    isPermissionDenied = true;
                }
            }
        }
    }

}
