/**
 * Copyright (C) 2003-2023, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdf.pdfviewer;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.os.StrictMode;
import android.provider.Settings;
import android.util.TypedValue;
import android.view.ActionMode;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Library;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.note.NoteModule;
import com.foxit.uiextensions.annots.textmarkup.highlight.HighlightModule;
import com.foxit.uiextensions.annots.textmarkup.squiggly.SquigglyModule;
import com.foxit.uiextensions.annots.textmarkup.strikeout.StrikeoutModule;
import com.foxit.uiextensions.annots.textmarkup.underline.UnderlineModule;
import com.foxit.uiextensions.controls.panel.PanelSpec;
import com.foxit.uiextensions.modules.SearchModule;
import com.foxit.uiextensions.modules.SearchView;
import com.foxit.uiextensions.modules.doc.docinfo.DocInfoModule;
import com.foxit.uiextensions.modules.doc.docinfo.DocInfoView;
import com.foxit.uiextensions.modules.pagenavigation.PageNavigationModule;
import com.foxit.uiextensions.modules.panel.annot.AnnotPanelModule;
import com.foxit.uiextensions.modules.panel.outline.OutlineModule;
import com.foxit.uiextensions.modules.thumbnail.ThumbnailModule;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentActivity;

public class MainActivity extends FragmentActivity {

    private PDFViewCtrl pdfViewCtrl = null;
    private UIExtensionsManager uiExtensionsManager = null;
    private SearchModule searchModule = null;
    private DocInfoModule docInfoModule = null;
    private NoteModule noteModule = null;
    private HighlightModule highlightModule = null;
    private PageNavigationModule pageNavigationModule = null;
    private UnderlineModule underlineModule = null;
    private StrikeoutModule strikeoutModule = null;
    private SquigglyModule squigglyModule = null;

    private AnnotPanelModule annotPanelModule;
    private OutlineModule outlineModule;
    private ThumbnailModule thumbnailModule;

    private boolean isUnlock = false;
    private Context mContext;
    private ActionMode mActionMode;

    private static String sn = "aKEpCRKu0/BkCmRFhavCoXOP9xI8QwJzIAEe4+CuldzMNJGzf4mKvQ==";
    private static String key = "ezJvjt3GpGp399sXJUWUjY5Q4Cy/RvybylNKusfzMufW5k1ziSNqlwk6IKRg++yKjh3VA44Jx0t3zTngokyX78qqdQX2n9Dq3DEulcv/f5XN1B6EX2jESCT4K53PBEncOAqvgam7qmIxuVj0yIJMcVzRju9BID1xQKOFHiRSUUXupPrEKTraVBJgzsphlgudAsQAk3xS56wpFAEAMJE9RhgHlzLd758GK5viOEglYUHl7n31UilbrvpOKiGEfQ52D54IwAofHdHamOMSlSYXhh2/JXWNKi+EMyegfB26Qg04+sry01n6SqLofbxC+6WtcRuwEj2I1Yb0lc6yFESdJWPGv7Dgz0COyGQbM6GDTNE33vaeFIa4MQdnKZuXABj9kGmTjeiZbcy8vkY7Lmm0gmnVu09VIpqZhYZwJM/aYdBRMPxBnKDGVw1t7hLD8jeOJAZX1Ua1jKjsLcni7s5+W7WU0sGKr7DJTwsEzcLqyr160onAaU3/M8ZQs7wy43rZ//oPYpx/mZHJs927TP7oVeRHx0sffTGpKPN7lctjA2hStQTkhCPl7nZhEz7c5EC3t9GJ+z4l5Sf+XLDILXyVTBkfDX897CFt1ozEU46y5TokHjpufDXSZxxBGcLxOiMfbW2MzfiB/BTL6/rbn/6kYM4aoGxhP4xWYEcVTpipDTzXlJ062+f4FEFoEt1YS5ufFA2v2nc9pqAzMpkciPXID6Wn822/PW/8cOu/bzVC1CH9lCG6YQtDV0PQyrjiOWkRQMN19N1Mfwooay7gA59NL01iSNYn5qx8R0O3oH2UfFBde3wCSqo+08ZvLnVPV0HJjacNHsuRlQvnGmBJvkNoeL/zb4O5t15dFBCrBUGRdlqTQl22r1n3sTNVoCkQfCGtY5wI/UbsCRWsjxphWBa0Edh1COD62UeY7Gq6oWy7O5b4kRjXVfMfc0q848KgUxfB03QFw7H40lHXODQOyKGfcCGMNdvA8zYRqSgg35DH9pmDqDB2rFyohlXy1fR+3e/Fud5tdlosppauZ/6rbFPiM5k4IAPQGRs0P/hW9MCeMZR8xnhcXO6Dabfnc7cdfkZqt+CI/+U0Jy+piWwCTnq97r5hba1Ng6eaCuvK9kzirBX1+HyQ+Fc6PEnIdi/lQNm7XjLy7aoWiUuNYoCFLHwdxV/XgGNyNALGz+HmHpESRMjjOjlhPIahkl9ENAZKW3Oq+k+K6d/fsht4ACz4sWSgTmiY+odH3OiwZ8wIhio0onqea9F9sHfWtHUcP+UYgxjVrH+/b32KhWaxwrtKiIg=";

    private static final int REQUEST_EXTERNAL_STORAGE = 111;
    private static final int REQUEST_ALL_FILES_ACCESS_PERMISSION = 222;
    private static final String[] PERMISSIONS_STORAGE = {
            Manifest.permission.READ_EXTERNAL_STORAGE,
            Manifest.permission.WRITE_EXTERNAL_STORAGE
    };

    private String getStorageDirectory() {
        String path = null;
        boolean sdExist = Environment.getExternalStorageState().equals(android.os.Environment.MEDIA_MOUNTED);
        if (sdExist) {
            path = Environment.getExternalStorageDirectory().getAbsolutePath() + "/";
        }
        return path;
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mContext = this;
        this.requestWindowFeature(Window.FEATURE_NO_TITLE);
        getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);

        int errorCode = Library.initialize(sn, key);
        isUnlock = true;

        if (errorCode != Constants.e_ErrSuccess) {
            isUnlock = false;

            String errorMsg = (errorCode == Constants.e_ErrInvalidLicense) ? getString(R.string.fx_the_license_is_invalid) : getString(R.string.fx_failed_to_initialize_the_library);
            UIToast.getInstance(getApplicationContext()).show(errorMsg);
            return;
        }

        pdfViewCtrl = new PDFViewCtrl(this.getApplicationContext());
        pdfViewCtrl.setAttachedActivity(this);
        pdfViewCtrl.registerDoubleTapEventListener(new PDFViewCtrl.IDoubleTapEventListener() {
            @Override
            public boolean onSingleTapConfirmed(MotionEvent e) {
                if (uiExtensionsManager.getState() != ReadStateConfig.STATE_NORMAL) {
                    return false;
                }
                if (pageNavigationModule == null)
                    pageNavigationModule = (PageNavigationModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_PAGENAV);

                if (mActionMode == null) {
                    createActionMode();

                    if (pageNavigationModule != null)
                        pageNavigationModule.changPageNumberState(true);
                } else {
                    mActionMode.finish();
                    if (pageNavigationModule != null)
                        pageNavigationModule.changPageNumberState(false);
                }
                return true;
            }


            @Override
            public boolean onDoubleTap(MotionEvent e) {
                return false;
            }

            @Override
            public boolean onDoubleTapEvent(MotionEvent e) {
                return false;
            }
        });

        uiExtensionsManager = new UIExtensionsManager(this.getApplicationContext(), pdfViewCtrl);
        uiExtensionsManager.enableBottomToolbar(false);
        uiExtensionsManager.enableTopToolbar(false);
        uiExtensionsManager.getConfig().uiSettings.fullscreen = false;
        uiExtensionsManager.setContinueAddAnnot(false);

        uiExtensionsManager.setAttachedActivity(this);
        uiExtensionsManager.onCreate(this, pdfViewCtrl, savedInstanceState);
        pdfViewCtrl.setUIExtensionsManager(uiExtensionsManager);
        uiExtensionsManager.registerMenuEventListener(mMenuEventListener);
        uiExtensionsManager.registerStateChangeListener(mStateChangeListener);

        outlineModule = (OutlineModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_OUTLINE);
        if (outlineModule == null) {
            outlineModule = new OutlineModule(this, uiExtensionsManager.getRootView(), pdfViewCtrl, uiExtensionsManager);
            outlineModule.loadModule();
        }
        annotPanelModule = (AnnotPanelModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_ANNOTPANEL);
        if (annotPanelModule == null) {
            annotPanelModule = new AnnotPanelModule(mContext, pdfViewCtrl, uiExtensionsManager);
            annotPanelModule.loadModule();
        }
        thumbnailModule = (ThumbnailModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_THUMBNAIL);
        if (thumbnailModule == null) {
            thumbnailModule = new ThumbnailModule(mContext, pdfViewCtrl, uiExtensionsManager);
            thumbnailModule.loadModule();
        }

        if (Build.VERSION.SDK_INT >= 24) {
            StrictMode.VmPolicy.Builder builder = new StrictMode.VmPolicy.Builder();
            StrictMode.setVmPolicy(builder.build());
        }

        View panelView = uiExtensionsManager.getRootView().findViewById(com.foxit.uiextensions.R.id.read_panel_view_ly);
        if (AppDisplay.isPad() && panelView != null && panelView.getLayoutParams() instanceof RelativeLayout.LayoutParams) {
            ((RelativeLayout.LayoutParams) panelView.getLayoutParams()).removeRule(RelativeLayout.BELOW);
            panelView.setLayoutParams(panelView.getLayoutParams());
        }
        setContentView(uiExtensionsManager.getContentView());

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            if (!Environment.isExternalStorageManager()) {
                Intent intent = new Intent(Settings.ACTION_MANAGE_APP_ALL_FILES_ACCESS_PERMISSION);
                intent.setData(Uri.parse("package:" + getApplicationContext().getPackageName()));
                startActivityForResult(intent, REQUEST_ALL_FILES_ACCESS_PERMISSION);
                return;
            }
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
            int permission = ContextCompat.checkSelfPermission(this.getApplicationContext(), Manifest.permission.WRITE_EXTERNAL_STORAGE);
            if (permission != PackageManager.PERMISSION_GRANTED) {
                ActivityCompat.requestPermissions(this, PERMISSIONS_STORAGE, REQUEST_EXTERNAL_STORAGE);
                return;
            }
        }
        openDocument();
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == REQUEST_EXTERNAL_STORAGE) {
            if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                openDocument();
            } else {
                UIToast.getInstance(getApplicationContext()).show("Permission Denied");
                finish();
            }
        }
    }

    private void openDocument() {
        // Note: Here, filePath will be set with the total path of file.
        AppThreadManager.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                String sdcardPath = getStorageDirectory();
                String filePath = sdcardPath + "FoxitSDK/Sample.pdf";
                uiExtensionsManager.openDocument(filePath, null);
            }
        });
    }

    @Override
    public void onStart() {
        if (uiExtensionsManager != null) {
            uiExtensionsManager.onStart(this);
        }
        super.onStart();
    }

    @Override
    public void onStop() {
        if (uiExtensionsManager != null) {
            uiExtensionsManager.onStop(this);
        }
        super.onStop();
    }

    @Override
    public void onPause() {
        if (uiExtensionsManager != null) {
            uiExtensionsManager.onPause(this);
        }
        super.onPause();
    }

    @Override
    public void onResume() {
        if (uiExtensionsManager != null) {
            uiExtensionsManager.onResume(this);
        }
        super.onResume();
    }

    @Override
    protected void onDestroy() {
        if (uiExtensionsManager != null) {
            uiExtensionsManager.unregisterMenuEventListener(mMenuEventListener);
            uiExtensionsManager.unregisterStateChangeListener(mStateChangeListener);
            uiExtensionsManager.onDestroy(this);
        }
        freeMemory();
        super.onDestroy();
    }

    private void freeMemory() {
        System.runFinalization();
        Runtime.getRuntime().gc();
        System.gc();
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (uiExtensionsManager != null) {
            if (mActionMode == null) {
                pdfViewCtrl.offsetScrollBoundary(0, 0, 0, 0);
                pdfViewCtrl.postPageContainer();
            } else {
                TypedValue tv = new TypedValue();
                int actionBarHeight;
                if (getTheme().resolveAttribute(android.R.attr.actionBarSize, tv, true)) {
                    actionBarHeight = TypedValue.complexToDimensionPixelSize(tv.data, getResources().getDisplayMetrics());
                    pdfViewCtrl.offsetScrollBoundary(0, actionBarHeight, 0, 0);
                    pdfViewCtrl.postPageContainer();
                }
            }
            uiExtensionsManager.onConfigurationChanged(this, newConfig);
        }
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (uiExtensionsManager != null && uiExtensionsManager.onKeyDown(this, keyCode, event))
            return true;
        return super.onKeyDown(keyCode, event);
    }

    private ActionMode.Callback mActionModeCallback = new ActionMode.Callback() {
        @Override
        public boolean onCreateActionMode(ActionMode mode, Menu menu) {
            MenuInflater inflater = mode.getMenuInflater();
            inflater.inflate(R.menu.main, menu);
            return true;
        }

        @Override
        public boolean onPrepareActionMode(ActionMode mode, Menu menu) {
            return false;
        }

        @Override
        public boolean onActionItemClicked(ActionMode mode, MenuItem item) {
            if (!isUnlock) {
                UIToast.getInstance(getApplicationContext()).show(getString(R.string.fx_unlock_library_failed));
                return false;
            }

            showSystemUI();
            uiExtensionsManager.triggerDismissMenuEvent();
            int itemId = item.getItemId();
            if (itemId == R.id.Note || itemId == R.id.Highlight
                    || itemId == R.id.Squiggly || itemId == R.id.Underline
                    || itemId == R.id.StrikeOut) {
                if (!uiExtensionsManager.canAddAnnot()) {
                    UIToast.getInstance(getApplicationContext()).show(getString(R.string.fx_the_document_cannot_modify));
                    return false;
                }
            }

            if (itemId == R.id.Outline) {
                if (outlineModule != null)
                    uiExtensionsManager.getPanelManager().showPanel(PanelSpec.OUTLINE);
            } else if (itemId == R.id.ChangeLayout) {
                pdfViewCtrl.setContinuous(!pdfViewCtrl.isContinuous());
                pdfViewCtrl.setPageLayoutMode(PDFViewCtrl.PAGELAYOUTMODE_SINGLE);
                PageNavigationModule pageNavigationModule = (PageNavigationModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_PAGENAV);
                if (pageNavigationModule != null) {
                    pageNavigationModule.resetJumpView();
                }
            } else if (itemId == R.id.Search) {
                if (searchModule == null) {

                    searchModule = (SearchModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_SEARCH);
                    if (searchModule == null) {
                        searchModule = new SearchModule(mContext, uiExtensionsManager.getRootView(), pdfViewCtrl, uiExtensionsManager);
                        searchModule.loadModule();
                    }
                }
                SearchView searchView = searchModule.getSearchView();
                searchView.show();
            } else if (itemId == R.id.Note) {
                if (noteModule == null) {
                    noteModule = (NoteModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_NOTE);
                }
                uiExtensionsManager.setCurrentToolHandler(noteModule.getToolHandler());
            } else if (itemId == R.id.DocInfo) {
                if (docInfoModule == null) {
                    docInfoModule = (DocInfoModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_DOCINFO);
                }

                DocInfoView docInfoView = docInfoModule.getView();
                if (docInfoView != null)
                    docInfoView.show();
            } else if (itemId == R.id.Highlight) {
                if (highlightModule == null)
                    highlightModule = (HighlightModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_HIGHLIGHT);
                uiExtensionsManager.setCurrentToolHandler(highlightModule.getToolHandler());
            } else if (itemId == R.id.Underline) {
                if (underlineModule == null) {
                    underlineModule = (UnderlineModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDERLINE);
                }
                uiExtensionsManager.setCurrentToolHandler(underlineModule.getToolHandler());
            } else if (itemId == R.id.StrikeOut) {
                if (strikeoutModule == null) {
                    strikeoutModule = (StrikeoutModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_STRIKEOUT);
                }
                uiExtensionsManager.setCurrentToolHandler(strikeoutModule.getToolHandler());
            } else if (itemId == R.id.Squiggly) {
                if (squigglyModule == null) {
                    squigglyModule = (SquigglyModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_SQUIGGLY);
                }
                uiExtensionsManager.setCurrentToolHandler(squigglyModule.getToolHandler());
            } else if (itemId == R.id.Annotations) {
                if (annotPanelModule != null)
                    uiExtensionsManager.getPanelManager().showPanel(PanelSpec.ANNOTATIONS);
            } else if (itemId == R.id.Thumbnail) {
                if (thumbnailModule != null) {
                    thumbnailModule.show();
                }
            }

            mode.finish();
            return true;
        }

        @Override
        public void onDestroyActionMode(ActionMode mode) {
            if (mActionMode != null)
                mActionMode = null;
            pdfViewCtrl.offsetScrollBoundary(0, 0, 0, 0);
            pdfViewCtrl.postPageContainer();
        }
    };

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == REQUEST_ALL_FILES_ACCESS_PERMISSION) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                if (Environment.isExternalStorageManager()) {
                    openDocument();
                }
            }
        } else {
            if (pdfViewCtrl != null) {
                pdfViewCtrl.handleActivityResult(requestCode, resultCode, data);
            }
        }
    }

    private IStateChangeListener mStateChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            boolean readState = newState == ReadStateConfig.STATE_REFLOW
                    || newState == ReadStateConfig.STATE_PANZOOM
                    || newState == ReadStateConfig.STATE_SEARCH
                    || newState == ReadStateConfig.STATE_TTS
                    || newState == ReadStateConfig.STATE_AUTOFLIP
                    || newState == ReadStateConfig.STATE_REDACT;
            if (readState) {
                if (mActionMode != null) {
                    mActionMode.finish();
                }
            } else {
                if (oldState != newState) {
                    if (mActionMode == null) {
                        createActionMode();
                    }
                }
            }
        }
    };

    private UIExtensionsManager.MenuEventListener mMenuEventListener = new UIExtensionsManager.MenuEventListener() {
        @Override
        public void onTriggerDismissMenu() {
            if (mActionMode != null)
                mActionMode.finish();
        }
    };

    private void createActionMode() {
        TypedValue tv = new TypedValue();
        int actionBarHeight;
        if (getTheme().resolveAttribute(android.R.attr.actionBarSize, tv, true)) {
            actionBarHeight = TypedValue.complexToDimensionPixelSize(tv.data, getResources().getDisplayMetrics());
            pdfViewCtrl.offsetScrollBoundary(0, actionBarHeight, 0, 0);
            pdfViewCtrl.postPageContainer();
        }
        mActionMode = ((Activity) mContext).startActionMode(mActionModeCallback);
        int doneButtonId = Resources.getSystem().getIdentifier("action_mode_close_button", "id", "android");
        View closeButton = findViewById(doneButtonId);
        if (closeButton != null) {
            closeButton.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    uiExtensionsManager.triggerDismissMenuEvent();
                    uiExtensionsManager.backToPrevActivity();
                }
            });
        }
    }

    // Shows the system bars by removing all the flags
    // except for the ones that make the content appear under the system bars.
    public void showSystemUI() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            getWindow().getDecorView().setSystemUiVisibility(
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                            | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN);
        }
    }

}
