import express from 'express';
import fileUpload from 'express-fileupload';
import path from 'path';
import fs from 'fs';
import cors from 'cors';

const FILE_UPLOAD_BASE = process.env['FILE_UPLOAD_BASE'] || path.resolve(process.cwd(), 'file-uploads')
const FILE_PATH_PREFIX = '/files'

// Whitelist: Only allow PDF files
const ALLOWED_MIME_TYPES = ['application/pdf'];
const ALLOWED_EXTENSIONS = ['.pdf'];

// PDF magic bytes for content validation
const PDF_MAGIC_BYTES = '25504446'; // %PDF in hex

/**
 * Validate if file is a genuine PDF by checking magic bytes
 */
function isPDFFile(buffer: Buffer): boolean {
  const fileHeader = buffer.slice(0, 4).toString('hex');
  return fileHeader === PDF_MAGIC_BYTES;
}

/**
 * Sanitize username to prevent path traversal and XSS attacks
 * Supports international characters (Chinese, Japanese, Korean, etc.)
 * Uses blacklist approach to exclude only unsafe characters
 */
function sanitizeUsername(username: string): string | null {
  if (!username || typeof username !== 'string') {
    return null;
  }
  
  // Reject usernames that are too long or reserved names
  if (username.length > 50 || username === '.' || username === '..') {
    return null;
  }
  
  // Reject usernames containing unsafe characters (use blacklist approach to support Unicode/international characters)
  // Exclude: path separators (/ \), null byte, control characters, and problematic characters (< > : " | ? * & ; $ ` ' = + , @)
  const unsafeCharsRegex = /[\/\\:\*\?"<>\|&;$`'=+,@]/;
  const controlCharsRegex = /[\x00-\x1f\x7f]/;
  const whitespaceRegex = /\s/;
  
  if (unsafeCharsRegex.test(username) || controlCharsRegex.test(username) || whitespaceRegex.test(username)) {
    return null;
  }
  
  return username;
}

/**
 * Sanitize filename to prevent path traversal and malicious filenames
 */
function sanitizeFilename(filename: string): string | null {
  if (!filename || typeof filename !== 'string') {
    return null;
  }
  // Get basename to remove any path components
  const basename = path.basename(filename);
  
  // Reject filenames that are too long or reserved names
  if (basename.length > 255 || basename === '.' || basename === '..') {
    return null;
  }
  
  // Reject filenames containing unsafe characters (use blacklist approach to support Unicode/international characters)
  // Exclude: path separators (/ \), null byte, control characters, and problematic characters (< > : " | ? *)
  const unsafeCharsRegex = /[\/\\:\*\?"<>\|]/;
  const controlCharsRegex = /[\x00-\x1f\x7f]/;
  
  if (unsafeCharsRegex.test(basename) || controlCharsRegex.test(basename)) {
    return null;
  }
  
  return basename;
}

/**
 * HTML escape function to prevent XSS attacks
 */
function escapeHtml(text: string): string {
  const map: { [key: string]: string } = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#039;'
  };
  return text.replace(/[&<>"']/g, (m) => map[m]);
}

function encodeFileName(filename: string){
  const t = path.parse(filename)
  const b = Buffer.from(t.name, 'utf8');
  return b.toString('hex') + t.ext;
}

function decodeFileName(filename: string){
  const t = path.parse(filename)
  const b = Buffer.from(t.name, 'hex');
  return b.toString('utf8') + t.ext;
}


/**
 * This is a simple file upload service based on Express which will be integrated to the sample server.
 */
export function setupFileService(app: any){

  console.log("initiated file service");
  app.use(cors({
    exposedHeaders: ['Content-Range', 'Content-Length', 'Accept-Ranges']
  }))
  app.use(fileUpload({
    createParentPath: true,
    defParamCharset: 'utf8'
  }))

  app.post("/api/files/upload", function(req: any, res: any){
    console.log("start uploading files")

    let uploadFile;
    let uploadPath;

    if (!req.files || Object.keys(req.files).length === 0) {
      return res.status(400).json({
        ret: -1,
        message: 'No files were uploaded.'
      });
    }

    // Validate and sanitize username
    const username = sanitizeUsername(req.query.username);
    if (!username) {
      return res.status(400).json({
        ret: -1,
        message: 'Invalid username parameter.'
      });
    }

    uploadFile = req.files.file;
    
    // Validate and sanitize filename
    const sanitizedOriginalName = sanitizeFilename(uploadFile.name);
    if (!sanitizedOriginalName) {
      return res.status(400).json({
        ret: -1,
        message: 'Invalid filename. Only alphanumeric characters, dots, underscores and hyphens are allowed.'
      });
    }

    // First, validate file content is actually a PDF (magic bytes check - most reliable)
    if (!isPDFFile(uploadFile.data)) {
      return res.status(400).json({
        ret: -1,
        message: 'Only PDF files are allowed. File content validation failed.'
      });
    }

    // If file has an extension, it must be .pdf
    const fileExtension = path.extname(sanitizedOriginalName).toLowerCase();
    if (fileExtension && !ALLOWED_EXTENSIONS.includes(fileExtension)) {
      return res.status(400).json({
        ret: -1,
        message: 'Only PDF files are allowed. Invalid file extension.'
      });
    }

    // Check MIME type - only allow PDF
    if (!ALLOWED_MIME_TYPES.includes(uploadFile.mimetype)) {
      return res.status(400).json({
        ret: -1,
        message: 'Only PDF files are allowed. Invalid MIME type.'
      });
    }

    const fileName = encodeFileName(sanitizedOriginalName);
    uploadPath = path.resolve(FILE_UPLOAD_BASE, username, fileName);

    // Verify the resolved path is within the expected base directory (prevent path traversal)
    const normalizedPath = path.normalize(uploadPath);
    const baseDir = path.resolve(FILE_UPLOAD_BASE, username);
    if (!normalizedPath.startsWith(baseDir)) {
      return res.status(400).json({
        ret: -1,
        message: 'Invalid file path detected.'
      });
    }

    console.log("saving files to: ", uploadPath)

    // Use the mv() method to place the file somewhere on your server
    uploadFile.mv(uploadPath, (err) => {
      if (err){
        return res
          .status(500)
          .json({
            ret: -1,
            message: err.message
          })
      }
      res.json({
        ret: 0,
        message: 'success',
        data: {
          path: `${FILE_PATH_PREFIX}/${fileName}`
        }
      });
    });
  })

  app.get("/api/files/list", function(req: any, res: any){
    // Validate and sanitize username
    const username = sanitizeUsername(req.query.username);
    if (!username) {
      return res.status(400).json({
        ret: -1,
        message: 'Invalid username parameter.'
      });
    }

    const userDir = path.resolve(FILE_UPLOAD_BASE, username);
    
    // Verify the resolved path is within the expected base directory (prevent path traversal)
    const normalizedDir = path.normalize(userDir);
    if (!normalizedDir.startsWith(path.resolve(FILE_UPLOAD_BASE))) {
      return res.status(400).json({
        ret: -1,
        message: 'Invalid directory path detected.'
      });
    }

    fs.readdir(userDir, (err, files) => {
      if(err || !files){
        res.json({
          ret: 0,
          data: []
        })
      }else{
        res.json({
          ret: 0,
          data: files.map(fileName => {
            const decodedName = decodeFileName(fileName);
            return {
              // HTML escape to prevent XSS attacks
              name: escapeHtml(decodedName),
              path: escapeHtml(`${FILE_PATH_PREFIX}/${username}/${fileName}`)
            }
          })
        })
      }
    })
  })

  app.use(FILE_PATH_PREFIX, express.static(path.resolve(FILE_UPLOAD_BASE), {
    // @ts-ignore
    acceptRanges: true,
    cacheControl: false,
    etag: false,
    lastModified: false
  }));
}
